/*:
 * @target MZ
 * @plugindesc v3.0 PKD Spine2D：push直前スナップ＋Map復帰で完全復元 & ロード成功後の強制レジューム
 * @author HS
 *
 * @help
 * ■やること
 * 1) Map→(Options/Save/Load/Title) へ遷移する直前に現在のSpine状態をスナップ。
 * 2) キャンセルでMapに戻ったときはSavedataから完全復元（表情/スキン/座標/スケール等）。
 * 3) ロード成功でMapへ遷移したときも「再生だけ」補正（timeScaleやトラック未設定を蹴り起こし）。
 *
 * @param waitFrames
 * @text 復元待ちフレーム（キャンセル戻り）
 * @type number
 * @min 0 @max 300
 * @default 12
 *
 * @param resumeOnLoadSuccess
 * @text ロード成功後も再生を補正
 * @type boolean
 * @default true
 *
 * @param resumeWaitFrames
 * @text ロード成功→補正までの待ち
 * @type number
 * @min 0 @max 300
 * @default 12
 *
 * @param resumeFallbackName
 * @text 補正時のデフォルトアニメ名（未取得時）
 * @type string
 * @default
 *
 * @param resumeFallbackLoop
 * @text デフォルトアニメをループ
 * @type boolean
 * @default true
 *
 * @param debugLog
 * @text デバッグログ
 * @type boolean
 * @default false
 */
(() => {
  "use strict";
  const PN = "HS_SpineSnapshotRestore";
  const P  = PluginManager.parameters(PN);
  const WAIT_RESTORE  = Math.max(0, Number(P.waitFrames || 12));
  const RESUME_ON_LOAD= String(P.resumeOnLoadSuccess || "true") === "true";
  const WAIT_RESUME   = Math.max(0, Number(P.resumeWaitFrames || 12));
  const FALLBACK_NAME = String(P.resumeFallbackName || "");
  const FALLBACK_LOOP = String(P.resumeFallbackLoop || "true") === "true";
  const DEBUG = String(P.debugLog || "false") === "true";
  const log   = (...a)=>{ if (DEBUG) console.log(`[${PN}]`, ...a); };

  const sys = ()=> $gameSystem;
  const tmp = ()=> $gameTemp;

  function initFlags(){
    if (!sys()) return;
    // スナップ→復元用
    sys()._hsSnapData ??= null;       // { id: savedataCopy, ... }
    sys()._hsPendRestore ??= false;   // キャンセル戻りの復元待ち
    sys()._hsRestoreWait ??= 0;
    // 成功フラグ
    sys()._hsLastSaveOK ??= false;
    sys()._hsLastLoadOK ??= false;
    // ロード成功後のレジューム
    sys()._hsPendResumeAfterLoad ??= false;
    sys()._hsResumeWait ??= 0;
  }

  // $gameTemp が持つ現在のSpine辞書
  const dict = ()=> (tmp() && tmp()._pSpineAnimations) || null;

  // PKDの保存倉庫
  const mgr = ()=> (window.PKD_SpineAnimations && window.PKD_SpineAnimations.Instance && window.PKD_SpineAnimations.Instance()) || null;
  const getSavedata = (id)=> {
    try { const m = mgr(); return m && m.getAnimationData ? m.getAnimationData(id) : null; }
    catch(e){ console.warn(e); return null; }
  };

  const deepCopy = (o)=> JSON.parse(JSON.stringify(o));

  // 現在のトラックから [name, loop, mixDuration, trackIndex] を抽出
  function makeStateFromTracks(sprite){
    try{
      const st = sprite.getAnimationState && sprite.getAnimationState();
      if (!st || !Array.isArray(st.tracks)) return null;
      const entry = st.tracks.find(t => t && t.animation);
      if (!entry || !entry.animation) return null;
      const name = entry.animation.name;
      const loop = !!entry.loop;
      const mix  = entry.mixDuration || 0;
      const idx  = entry.trackIndex || 0;
      return [name, loop, mix, idx];
    }catch(e){ console.warn(e); return null; }
  }

  // ---- push直前でスナップ（Map.stopより前）
  function snapshotBeforePush(){
    initFlags();
    const d = dict();
    if (!d) { log("snapshot: no temp dict"); return; }
    const ids = Object.keys(d).filter(k => !!d[k]);
    if (!ids.length){ log("snapshot: no spines"); return; }

    // 最新状態を倉庫へ
    for (const id of ids){
      try { const s = d[id]; if (s && typeof s.keep === "function") s.keep(); } catch(e){ console.warn(e); }
    }
    // 倉庫→コピー。欠損があれば補完
    const store = {};
    for (const id of ids){
      const saved = getSavedata(id);
      if (!saved) continue;
      const sCopy = deepCopy(saved);

      // timeScale補正
      if (!(typeof sCopy.timeScale === "number") || !(sCopy.timeScale > 0)) {
        try {
          const st = d[id].getAnimationState && d[id].getAnimationState();
          sCopy.timeScale = (st && st.timeScale > 0) ? st.timeScale : 1;
        } catch(e){ sCopy.timeScale = 1; }
      }
      // lastAnimationState補完
      if (!Array.isArray(sCopy.lastAnimationState) || sCopy.lastAnimationState.length === 0) {
        const fb = makeStateFromTracks(d[id]);
        if (fb) sCopy.lastAnimationState = fb;
      }
      store[id] = sCopy;
    }
    if (Object.keys(store).length){
      sys()._hsSnapData = store;
      log("snapshot: ids =", Object.keys(store));
    } else {
      log("snapshot: savedata none");
    }
  }

  // ---- キャンセル戻り：復元の予約
  function pendRestore(){
    initFlags();
    sys()._hsPendRestore = true;
    sys()._hsRestoreWait = WAIT_RESTORE;
    log("pend restore after", WAIT_RESTORE, "frames");
  }

  // ---- 復元実行（キャンセル戻り時）
  function doRestore(){
    initFlags();
    const store = sys()._hsSnapData;
    if (!store || !Object.keys(store).length){ log("restore: no ids"); return; }

    const Klass = window.PKD_Sprite_SpineAnimation;
    const byId  = Klass && Klass.ByID;
    const from  = Klass && Klass.FromSaveData;

    for (const id of Object.keys(store)){
      const data = store[id];
      let spr = null;
      try {
        spr = byId ? byId(id) : null;
        if (spr && typeof spr.applySavedata === "function"){
          spr.applySavedata(data);
          log("applySavedata:", id);
        } else if (typeof from === "function"){
          spr = from(data);
          log("FromSaveData:", id);
        }
        if (!spr) continue;

        // timeScale補正
        try {
          const st = spr.getAnimationState && spr.getAnimationState();
          if (st && !(st.timeScale > 0)) st.timeScale = Math.max(1, data.timeScale || 1);
        } catch(e){}

        // トラック未設定なら「直前スナップ」→ sprite.lastAnimationState → 現在 → Fallback の順で再指示
        try {
          const st = spr.getAnimationState && spr.getAnimationState();
          const hasTrack = !!(st && Array.isArray(st.tracks) && st.tracks.some(t => t && t.animation));
          if (!hasTrack) {
            // 1) push直前に撮ったスナップ（id一致）を最優先
            const store = sys()._hsSnapData;
            const snap  = store && store[id];
            let arr = null;
            if (snap && Array.isArray(snap.lastAnimationState) && snap.lastAnimationState.length) {
              arr = snap.lastAnimationState;
              // timeScale が止まっていればスナップの値で起動（なければ1）
              if (st && !(st.timeScale > 0)) st.timeScale = Math.max(1, Number(snap.timeScale || 1));
            }

            // 2) それでも無ければ sprite 側の lastAnimationState
            if (!arr) arr = Array.isArray(spr.lastAnimationState) ? spr.lastAnimationState : null;

            // 3) まだ無ければ「現在のトラックから推定」
            if (!arr) arr = makeStateFromTracks(spr);

            // 4) 最後の砦として Fallback
            if (!arr && FALLBACK_NAME) arr = [FALLBACK_NAME, FALLBACK_LOOP, 0, 0];

            if (arr){
              const [name, loop, mix, idx] = [arr[0] ?? "", !!arr[1], Number(arr[2] ?? 0) || 0, Number(arr[3] ?? 0) || 0];
              spr.setAnimation(name, loop, mix, idx);
              log(snap ? "afterLoad(set-by-snapshot):" : "afterLoad(set):", id, arr);
            } else {
              log("afterLoad: no state", id);
            }
          }
        } catch(e){ console.warn(e); }

        // 自動更新ON
        try {
          const sp = spr.getSpineObject && spr.getSpineObject();
          if (sp && sp.autoUpdate === false) sp.autoUpdate = true;
          if (sp && sp.update) sp.update(0);
        } catch(e){}
      } catch(e){ console.warn(e); }
    }
    // 使い終わったら破棄
    sys()._hsSnapData = null;
  }

  // ---- ロード成功後：再生だけ補正
  function pendResumeAfterLoad(){
    if (!RESUME_ON_LOAD) return;
    initFlags();
    sys()._hsPendResumeAfterLoad = true;
    sys()._hsResumeWait = WAIT_RESUME;
    // ※このときスナップは使わない（ロード先の状態を優先）
    log("pend resume-after-load in", WAIT_RESUME, "frames");
  }

  function doResumeAfterLoad(){
    initFlags();
    const d = dict();
    if (!d) { log("resume: no dict"); return; }
    const ids = Object.keys(d).filter(k => !!d[k]);
    if (!ids.length){ log("resume: no spines"); return; }

    for (const id of ids){
      const spr = d[id];
      try {
        // timeScale>0 に
        try {
          const st = spr.getAnimationState && spr.getAnimationState();
          if (st && !(st.timeScale > 0)) st.timeScale = 1;
        } catch(e){}
        // トラック無ければ lastAnimationState / 現在 / Fallback で起動
        try {
          const st = spr.getAnimationState && spr.getAnimationState();
          const hasTrack = !!(st && Array.isArray(st.tracks) && st.tracks.some(t => t && t.animation));
          if (!hasTrack) {
            let arr = Array.isArray(spr.lastAnimationState) ? spr.lastAnimationState : makeStateFromTracks(spr);
            if (!arr && FALLBACK_NAME) arr = [FALLBACK_NAME, FALLBACK_LOOP, 0, 0];
            if (arr){
              const [name, loop, mix, idx] = [arr[0] ?? "", !!arr[1], Number(arr[2] ?? 0) || 0, Number(arr[3] ?? 0) || 0];
              spr.setAnimation(name, loop, mix, idx);
              log("afterLoad(set):", id, arr);
            } else {
              log("afterLoad: no state", id);
            }
          }
        } catch(e){ console.warn(e); }
        // 自動更新ON
        try {
          const sp = spr.getSpineObject && spr.getSpineObject();
          if (sp && sp.autoUpdate === false) sp.autoUpdate = true;
          if (sp && sp.update) sp.update(0);
        } catch(e){}
      } catch(e){ console.warn(e); }
    }
  }

  // ---- Save/Load 成功フラグ
  if (Scene_Save) {
    const _ok = Scene_Save.prototype.onSaveSuccess;
    Scene_Save.prototype.onSaveSuccess = function(){
      initFlags(); sys()._hsLastSaveOK = true; log("save success");
      _ok.call(this);
    };
  }
  if (Scene_Load) {
    const _ok = Scene_Load.prototype.onLoadSuccess;
    Scene_Load.prototype.onLoadSuccess = function(){
      initFlags(); sys()._hsLastLoadOK = true; log("load success");
      // ロード成功後はMapへ goto される → Map側で補正を走らせる
      pendResumeAfterLoad();
      _ok.call(this);
    };
  }

  // ---- SceneManager.push：Map上→対象シーンへ遷移する直前にスナップ
  const _push = SceneManager.push;
  SceneManager.push = function(sceneClass){
    try {
      const cur = this._scene;
      const target = sceneClass && sceneClass.name;
      const isMap = cur instanceof Scene_Map;
      const isTarget =
        target === "Scene_Options" ||
        target === "Scene_Save"    ||
        target === "Scene_Load"    ||
        target === "Scene_File"    ||
        target === "Scene_GameEnd";
      if (isMap && isTarget) snapshotBeforePush();
    } catch(e){ console.warn(e); }
    return _push.call(this, sceneClass);
  };

  // ---- 復帰判定（キャンセル＝復元 / ロード成功＝復元しない）
  if (Scene_Options) {
    const _pop = Scene_Options.prototype.popScene;
    Scene_Options.prototype.popScene = function(){
      try { pendRestore(); } catch(e){ console.warn(e); }
      _pop.call(this);
    };
  }
  if (Scene_File) {
    const _pop = Scene_File.prototype.popScene;
    Scene_File.prototype.popScene = function(){
      initFlags();
      const ok = (this instanceof Scene_Save) ? sys()._hsLastSaveOK :
                 (this instanceof Scene_Load) ? sys()._hsLastLoadOK : false;
      if (!ok) pendRestore();
      sys()._hsLastSaveOK = false;
      sys()._hsLastLoadOK = false;
      _pop.call(this);
    };
  }
  if (Scene_GameEnd) {
    const _cancel = Scene_GameEnd.prototype.commandCancel;
    if (_cancel) {
      Scene_GameEnd.prototype.commandCancel = function(){
        try { pendRestore(); } catch(e){ console.warn(e); }
        return _cancel.call(this);
      };
    }
    const _pop = Scene_GameEnd.prototype.popScene;
    if (_pop) {
      Scene_GameEnd.prototype.popScene = function(){
        try { pendRestore(); } catch(e){ console.warn(e); }
        return _pop.call(this);
      };
    }
  }

  // ---- Map側：復元 / ロード補正 を遅延実行
  const _mapStart = Scene_Map.prototype.start;
  Scene_Map.prototype.start = function(){
    _mapStart.call(this);
    initFlags();
    this._hsRestoreWait = sys()._hsPendRestore        ? (sys()._hsRestoreWait || 0) : -1;
    this._hsResumeWait  = sys()._hsPendResumeAfterLoad? (sys()._hsResumeWait  || 0) : -1;
    if (this._hsRestoreWait >= 0) log("restore in", this._hsRestoreWait, "frames");
    if (this._hsResumeWait  >= 0) log("afterLoad resume in", this._hsResumeWait, "frames");
  };
  const _mapUpdate = Scene_Map.prototype.update;
  Scene_Map.prototype.update = function(){
    _mapUpdate.call(this);

    if (this._hsRestoreWait != null && this._hsRestoreWait >= 0){
      if (this._hsRestoreWait-- <= 0){
        try { doRestore(); } catch(e){ console.warn(e); }
        sys()._hsPendRestore = false;
        sys()._hsRestoreWait = 0;
        this._hsRestoreWait = -1;
      }
    }
    if (this._hsResumeWait != null && this._hsResumeWait >= 0){
      if (this._hsResumeWait-- <= 0){
        try { doResumeAfterLoad(); } catch(e){ console.warn(e); }
        sys()._hsPendResumeAfterLoad = false;
        sys()._hsResumeWait = 0;
        this._hsResumeWait = -1;
      }
    }
  };
})();

